#!/bin/bash

# OpenVPN 客户端自动安装配置脚本（包含仓库修复）
# 使用方法: sudo ./vpn.sh

set -e  # 遇到错误立即退出

echo "=== OpenVPN 客户端自动安装配置脚本（包含仓库修复）==="

# 检查是否以root权限运行
if [ "$EUID" -ne 0 ]; then
    echo "请使用 sudo 运行此脚本"
    exit 1
fi

# 获取当前脚本所在目录
CURRENT_DIR=$(pwd)
echo "当前目录: $CURRENT_DIR"

# 创建客户端配置目录（提前创建，以便后续使用）
mkdir -p /etc/openvpn/client

# 检查标记文件，判断是否已经执行过仓库修复
REPO_FIXED_FILE="/etc/openvpn/client/.repo_fixed"

# 第一部分：修复 CentOS 仓库（仅在第一次运行时执行）
if [ ! -f "$REPO_FIXED_FILE" ]; then
    echo "1. 首次运行，修复 CentOS-Base.repo 文件..."
    
    # 进入目录
    cd /etc/yum.repos.d

    # 备份原文件
    if [ -f "CentOS-Base.repo" ]; then
        cp CentOS-Base.repo CentOS-Base.repo.backup
    fi

    # 创建新的 CentOS-Base.repo 文件
    cat > CentOS-Base.repo << 'EOF2'
# CentOS-Base.repo
#
# The mirror system uses the connecting IP address of the client and the
# update status of each mirror to pick mirrors that are updated to and
# geographically close to the client.  You should use this for CentOS updates
# unless you are manually picking other mirrors.
#
# If the mirrorlist= does not work for you, as a fall back you can try the 
# remarked out baseurl= line instead.
#
#
 
[base]
name=CentOS-$releasever - Base
#mirrorlist=http://mirrorlist.centos.org/?release=$releasever&arch=$basearch&repo=os&infra=$infra
#baseurl=http://mirror.centos.org/centos/$releasever/os/$basearch/
#baseurl=http://vault.centos.org/7.9.2009/x86_64/os/
baseurl=http://vault.centos.org/7.9.2009/os/$basearch/
gpgcheck=1
gpgkey=file:///etc/pki/rpm-gpg/RPM-GPG-KEY-CentOS-7
 
#released updates 
[updates]
name=CentOS-$releasever - Updates
#mirrorlist=http://mirrorlist.centos.org/?release=$releasever&arch=$basearch&repo=updates&infra=$infra
#baseurl=http://mirror.centos.org/centos/$releasever/updates/$basearch/
#baseurl=http://vault.centos.org/7.9.2009/x86_64/os/
baseurl=http://vault.centos.org/7.9.2009/updates/$basearch/
gpgcheck=1
gpgkey=file:///etc/pki/rpm-gpg/RPM-GPG-KEY-CentOS-7
 
#additional packages that may be useful
[extras]
name=CentOS-$releasever - Extras
#mirrorlist=http://mirrorlist.centos.org/?release=$releasever&arch=$basearch&repo=extras&infra=$infra
#$baseurl=http://mirror.centos.org/centos/$releasever/extras/$basearch/
#baseurl=http://vault.centos.org/7.9.2009/x86_64/os/
baseurl=http://vault.centos.org/7.9.2009/extras/$basearch/
gpgcheck=1
gpgkey=file:///etc/pki/rpm-gpg/RPM-GPG-KEY-CentOS-7
 
#additional packages that extend functionality of existing packages
[centosplus]
name=CentOS-$releasever - Plus
#mirrorlist=http://mirrorlist.centos.org/?release=$releasever&arch=$basearch&repo=centosplus&infra=$infra
#baseurl=http://mirror.centos.org/centos/$releasever/centosplus/$basearch/
#baseurl=http://vault.centos.org/7.9.2009/x86_64/os/
baseurl=http://vault.centos.org/7.9.2009/centosplus/$basearch/
gpgcheck=1
enabled=0
gpgkey=file:///etc/pki/rpm-gpg/RPM-GPG-KEY-CentOS-7
EOF2

    # 清理并重建缓存
    echo "清理并重建YUM缓存..."
    yum clean all
    yum makecache

    echo "CentOS-Base.repo 已成功更新并缓存已重建"
    
    # 创建标记文件
    touch "$REPO_FIXED_FILE"
    echo "仓库修复完成，已创建标记文件: $REPO_FIXED_FILE"
    echo ""
else
    echo "1. 检测到已修复过仓库配置，跳过此步骤"
    echo ""
fi

# 第二部分：安装和配置OpenVPN（检查是否已安装）
echo "2. 检查并安装OpenVPN..."

# 检查OpenVPN是否已安装
if ! command -v openvpn &> /dev/null; then
    echo "OpenVPN未安装，开始安装..."
    yum -y install epel-release
    yum -y install openvpn
    echo "OpenVPN 安装完成！"
else
    echo "OpenVPN 已安装，版本: $(openvpn --version | head -1)"
fi

# 第三部分：复制配置文件
echo "3. 处理OpenVPN配置文件..."

# 检查当前目录是否有 OpenVPN-Client.conf 文件
if [ -f "$CURRENT_DIR/OpenVPN-Client.conf" ]; then
    # 检查目标文件是否存在，以及是否需要更新
    if [ -f "/etc/openvpn/client/OpenVPN-Client.conf" ]; then
        # 比较文件内容是否相同
        if cmp -s "$CURRENT_DIR/OpenVPN-Client.conf" "/etc/openvpn/client/OpenVPN-Client.conf"; then
            echo "配置文件已存在且内容相同，跳过复制"
        else
            echo "配置文件已存在但内容不同，正在更新..."
            cp "$CURRENT_DIR/OpenVPN-Client.conf" /etc/openvpn/client/
            chmod 600 /etc/openvpn/client/OpenVPN-Client.conf
            echo "配置文件已更新！"
        fi
    else
        echo "找到 $CURRENT_DIR/OpenVPN-Client.conf 文件，正在复制到 /etc/openvpn/client/"
        cp "$CURRENT_DIR/OpenVPN-Client.conf" /etc/openvpn/client/
        chmod 600 /etc/openvpn/client/OpenVPN-Client.conf
        echo "配置文件复制完成！"
    fi
    echo "原文件保留在: $CURRENT_DIR/OpenVPN-Client.conf"
else
    echo "错误：当前目录($CURRENT_DIR)没有找到 OpenVPN-Client.conf 文件"
    echo "请确保 OpenVPN-Client.conf 文件位于当前目录"
    echo "当前目录内容:"
    ls -la "$CURRENT_DIR/"
    exit 1
fi

# 第四部分：处理账号密码
AUTH_FILE="/etc/openvpn/client/auth.txt"

# 检查是否已有保存的账号密码
if [ -f "$AUTH_FILE" ]; then
    echo "4. 检测到已保存的账号密码文件"
    read -p "是否使用已保存的账号密码？(y/n): " use_saved
    if [ "$use_saved" = "y" ] || [ "$use_saved" = "Y" ]; then
        echo "使用已保存的账号密码"
        # 修改配置文件使用密码文件
        sed -i 's/^auth-user-pass$/auth-user-pass \/etc\/openvpn\/client\/auth.txt/' /etc/openvpn/client/OpenVPN-Client.conf
    else
        # 输入新的账号密码并保存
        read -p "请输入VPN用户名: " username
        read -s -p "请输入VPN密码: " password
        echo
        echo "$username" > "$AUTH_FILE"
        echo "$password" >> "$AUTH_FILE"
        chmod 600 "$AUTH_FILE"
        echo "账号密码已保存到 $AUTH_FILE"
        # 修改配置文件使用密码文件
        sed -i 's/^auth-user-pass$/auth-user-pass \/etc\/openvpn\/client\/auth.txt/' /etc/openvpn/client/OpenVPN-Client.conf
    fi
else
    # 第一次运行，输入并保存账号密码
    echo "4. 首次运行，请输入VPN账号密码"
    read -p "请输入VPN用户名: " username
    read -s -p "请输入VPN密码: " password
    echo
    echo "$username" > "$AUTH_FILE"
    echo "$password" >> "$AUTH_FILE"
    chmod 600 "$AUTH_FILE"
    echo "账号密码已保存到 $AUTH_FILE"
    # 修改配置文件使用密码文件
    sed -i 's/^auth-user-pass$/auth-user-pass \/etc\/openvpn\/client\/auth.txt/' /etc/openvpn/client/OpenVPN-Client.conf
fi

# 验证安装
echo "5. 验证安装..."
echo "OpenVPN 版本: $(openvpn --version | head -1)"
echo "配置文件位置: /etc/openvpn/client/OpenVPN-Client.conf"
echo "账号密码文件: /etc/openvpn/client/auth.txt"
echo "原配置文件保留在: $CURRENT_DIR/OpenVPN-Client.conf"

# 创建便捷管理脚本（如果不存在）
if [ ! -f "/usr/local/bin/vpn-manage" ]; then
    echo "6. 创建便捷管理脚本..."
    cat > /usr/local/bin/vpn-manage << 'EOF'
#!/bin/bash

case "$1" in
    "start")
        cd /etc/openvpn/client/
        openvpn --config OpenVPN-Client.conf
        ;;
    "stop")
        pkill openvpn
        echo "VPN已停止"
        ;;
    "status")
        if pgrep openvpn > /dev/null; then
            echo "VPN正在运行"
        else
            echo "VPN未运行"
        fi
        ;;
    "change-auth")
        read -p "请输入新的VPN用户名: " username
        read -s -p "请输入新的VPN密码: " password
        echo
        echo "$username" > /etc/openvpn/client/auth.txt
        echo "$password" >> /etc/openvpn/client/auth.txt
        chmod 600 /etc/openvpn/client/auth.txt
        echo "账号密码已更新"
        ;;
    "update-config")
        # 更新配置文件功能
        if [ -f "./OpenVPN-Client.conf" ]; then
            cp ./OpenVPN-Client.conf /etc/openvpn/client/
            chmod 600 /etc/openvpn/client/OpenVPN-Client.conf
            echo "配置文件已更新"
        else
            echo "错误：当前目录没有找到 OpenVPN-Client.conf 文件"
        fi
        ;;
    "reset-repo")
        # 重置仓库标记，下次运行会重新修复仓库
        rm -f /etc/openvpn/client/.repo_fixed
        echo "仓库标记已重置，下次运行将重新修复仓库"
        ;;
    *)
        echo "使用方法: vpn-manage {start|stop|status|change-auth|update-config|reset-repo}"
        echo "  start         - 启动VPN连接"
        echo "  stop          - 停止VPN连接"
        echo "  status        - 查看VPN状态"
        echo "  change-auth   - 修改账号密码"
        echo "  update-config - 更新配置文件（从当前目录复制）"
        echo "  reset-repo    - 重置仓库标记（下次重新修复仓库）"
        ;;
esac
EOF

    chmod +x /usr/local/bin/vpn-manage
    echo "便捷管理脚本创建完成！"
else
    echo "6. 便捷管理脚本已存在，跳过创建"
fi

echo ""
echo "=== 安装完成！==="
echo "使用说明:"
echo "启动VPN:     vpn-manage start"
echo "停止VPN:     vpn-manage stop"
echo "查看状态:    vpn-manage status"
echo "修改密码:    vpn-manage change-auth"
echo "更新配置:    vpn-manage update-config"
echo "重置仓库:    vpn-manage reset-repo"

# 询问是否立即启动VPN
read -p "是否立即启动VPN连接？(y/n): " start_now
if [ "$start_now" = "y" ] || [ "$start_now" = "Y" ]; then
    echo "正在启动OpenVPN连接..."
    cd /etc/openvpn/client/
    openvpn --config OpenVPN-Client.conf
else
    echo "您可以使用 'vpn-manage start' 命令随时启动VPN"
fi
