#!/bin/bash

# OpenVPN Systemd 服务设置脚本
# 使用方法: sudo ./setup-vpn-service.sh

set -e

echo "=== OpenVPN Systemd 服务设置脚本 ==="

# 检查root权限
if [ "$EUID" -ne 0 ]; then
    echo "请使用 sudo 运行此脚本"
    exit 1
fi

# 检查OpenVPN是否安装
if ! command -v openvpn &> /dev/null; then
    echo "错误：OpenVPN 未安装，请先安装OpenVPN"
    exit 1
fi

# 检查配置文件是否存在
if [ ! -f "/etc/openvpn/client/OpenVPN-Client.conf" ]; then
    echo "错误：OpenVPN 配置文件不存在"
    echo "请确保 /etc/openvpn/client/OpenVPN-Client.conf 文件存在"
    exit 1
fi

# 获取OpenVPN的实际路径
OPENVPN_PATH=$(which openvpn)
echo "检测到 OpenVPN 路径: $OPENVPN_PATH"

# 创建systemd服务文件
echo "创建systemd服务文件..."
cat > /etc/systemd/system/openvpn-client.service << EOF
[Unit]
Description=OpenVPN Client Service
After=network.target

[Service]
Type=simple
ExecStart=$OPENVPN_PATH --config /etc/openvpn/client/OpenVPN-Client.conf
WorkingDirectory=/etc/openvpn/client/
Restart=on-failure
RestartSec=5

[Install]
WantedBy=multi-user.target
EOF

echo "服务文件已创建: /etc/systemd/system/openvpn-client.service"

# 重新加载systemd配置
echo "重新加载systemd配置..."
systemctl daemon-reload

# 启用开机自启动
echo "启用开机自启动..."
systemctl enable openvpn-client

# 启动服务
echo "启动OpenVPN服务..."
systemctl start openvpn-client

# 等待一会儿让服务启动
sleep 3

# 查看服务状态
echo "查看服务状态:"
systemctl status openvpn-client --no-pager -l

# 显示服务日志
echo ""
echo "最近的服务日志:"
journalctl -u openvpn-client --since="1 minute ago" --no-pager

# 提供管理命令
echo ""
echo "=== 服务设置完成！ ==="
echo "管理命令:"
echo "启动服务:    systemctl start openvpn-client"
echo "停止服务:    systemctl stop openvpn-client"
echo "重启服务:    systemctl restart openvpn-client"
echo "查看状态:    systemctl status openvpn-client"
echo "查看日志:    journalctl -u openvpn-client -f"
echo "禁用自启动:  systemctl disable openvpn-client"

# 检查服务是否正常运行
if systemctl is-active --quiet openvpn-client; then
    echo ""
    echo "✅ OpenVPN 服务正在正常运行！"
else
    echo ""
    echo "❌ OpenVPN 服务启动失败，请检查以上日志信息"
    echo "可能的原因:"
    echo "1. OpenVPN 配置文件错误"
    echo "2. 认证信息不正确"
    echo "3. 网络连接问题"
    echo ""
    echo "请检查: journalctl -u openvpn-client -xe"
fi
